package gov.va.med.domain.service.messaging.decode.hl7;

import gov.va.med.domain.model.StatusPayload;
import gov.va.med.domain.model.validation.ValidationError;
import gov.va.med.domain.service.messaging.MessagingException;
import gov.va.med.domain.service.messaging.decode.MessageDecoder;
import gov.va.med.domain.service.messaging.encode.hl7.util.HL7Helper;
import gov.va.med.domain.service.messaging.parse.DefaultHL7ParseErrorListener;
import gov.va.med.domain.service.messaging.parse.DefaultHL7ParseEventListener;
import gov.va.med.domain.service.messaging.parse.HL7DecoderDataError;
import gov.va.med.domain.service.messaging.parse.HL7FilteringParser;
import gov.va.med.domain.service.messaging.parse.HL7MessageMetadata;
import gov.va.med.domain.service.messaging.parse.IHL7MessageClassOverrideListener;
import gov.va.med.domain.service.messaging.parse.IHL7ParseErrorListener;
import gov.va.med.domain.service.messaging.parse.IHL7ParseEventListener;
import gov.va.med.domain.service.messaging.transceiver.BaseHTTPTransceiver;

import java.math.BigDecimal;
import java.util.Date;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import ca.uhn.hl7v2.HL7Exception;
import ca.uhn.hl7v2.model.AbstractSegment;
import ca.uhn.hl7v2.model.GenericComposite;
import ca.uhn.hl7v2.model.Message;
import ca.uhn.hl7v2.model.Type;
import ca.uhn.hl7v2.model.Varies;
import ca.uhn.hl7v2.model.v24.datatype.CE;
import ca.uhn.hl7v2.model.v24.datatype.NM;
import ca.uhn.hl7v2.model.v24.datatype.ST;
import ca.uhn.hl7v2.model.v24.segment.ERR;
import ca.uhn.hl7v2.model.v24.segment.MSA;
import ca.uhn.hl7v2.parser.EncodingNotSupportedException;

/**
 * HL7MessageDecoder. Place holder for any common behavior or state needed by
 * HL7 decoders.
 * 
 * @author Slava Uchitel
 * @version $Id: HL7MessageDecoder.java,v 1.24 2005/08/24 19:35:12 tom Exp $
 * @since MHV 2.0 <br>
 *        Mar 1, 2005
 */
public abstract class HL7MessageDecoder extends MessageDecoder {
    private IHL7ParseErrorListener errorListener;
    private IHL7ParseEventListener eventListener;
    private IHL7MessageClassOverrideListener classOverrideListener;
    private HL7MessageMetadata metaData;
    
    private static Logger logger = LogManager.getLogger(HL7MessageDecoder.class);
    
    public HL7MessageDecoder() {
        initializeListeners();
        
    }
	/**
	 * Parses a String into an Hl7 Message. Logs the Excpetion.
	 *
	 * @param encodedPayload declared as Object but casted to a String
	 * @return the resulting Message
	 * @throws MessagingException wraps any exepction occurring.
	 */
	protected Message parse(Object encodedPayload) throws MessagingException {

	    HL7FilteringParser parser = new HL7FilteringParser(getExpectedHL7Version());
	    parser.setErrorListener(getErrorListener()); 
	    parser.setEventListener(getEventListener());
	    parser.setClassOverrideListener(getClassOverrideListener());
		try {
		    Message message =  parser.parse((String)encodedPayload);
		    return message;
		}
		catch(Exception e) {
			String message = "Failed to parse incoming message.";
			getLogger().error(message + " Message: \n" + encodedPayload, e);
			throw new MessagingException(message, e);
		}
	}
	/**
	 * Determines the appropriate ErrorListener and EventListener for 
	 * the target message type and sets it.    
	 */
	public void initializeListeners(){
	   setErrorListener(new DefaultHL7ParseErrorListener());
	   setEventListener(new DefaultHL7ParseEventListener());
	   setClassOverrideListener(
	       new IHL7MessageClassOverrideListener(){
	           public Class onFindMessageClass(Class messageClass, String structure, String version)
	           {
	               return null;
	           }
	       }
	   );
	}
	
	/**
	 * @param message
	 * @return payload containing error if not AA or null if AA
	 * @throws MessagingException
	 */
	protected StatusPayload checkMSAStatus(Message message)
	    throws MessagingException {
		try {
			MSA msaSegment = (MSA)message.get("MSA");
			if(msaSegment == null) {
				return null;
			}
			String ackCode = msaSegment.getAcknowledgementCode().getValue();
			if(ackCode.equalsIgnoreCase("AA")) {
				return null;
			}

			ERR errSegment = (ERR)message.get("ERR");
			if(errSegment == null) {
				return deriveErrorFromMSA(msaSegment);
			}
			return extractErrorStatus(errSegment);

		}

		catch(EncodingNotSupportedException e) {
			throw new MessagingException("Error processing incoming message: \r\n" + message, e);
		}
		catch(HL7Exception e) {
			throw new MessagingException("Error processing incoming message: \r\n" + message, e);
		}
	}
	/**
	 * Constructs a StatusPayload form an error segment (ERR) 
	 * @param errSegment from any HL7 message 
	 * @return a StatusPayload containg the error
	 * @throws HL7Exception 
	 */
	public StatusPayload extractErrorStatus(ERR errSegment) throws HL7Exception {
        String error = errSegment.getErrorCodeAndLocation(0).getCodeIdentifyingError().getText().getValue();
		return new StatusPayload(error);    
	}
	
	private StatusPayload deriveErrorFromMSA(MSA msaSegment) {
		return new StatusPayload(msaSegment.getErrorCondition().getText()
		                         .getValue());
	}
	
	protected Logger getLogger() {
        return logger;
    }
	
	protected BigDecimal getBigDecimal(AbstractSegment segment, int column, int rep, String fieldName) 
					throws HL7Exception{
	    String rawNumber = null; 
	    //(Varies)rdtSegment.getField(1,0)
	    try {
		    rawNumber = getString(segment, column, rep);
		    return new BigDecimal(rawNumber); 
	    }
	    catch (Exception e) {
	        HL7DecoderDataError error = new HL7DecoderDataError();
	        error.setErrorKey(ValidationError.INVALID_NUMBER_ERROR);
	        error.setComponentNumber(rep);
	        error.setFieldNumber(column);
	        error.setValue(rawNumber);
	        error.setException(new HL7Exception(e));  
	        error.setStructure(segment);
	        error.setDataType("Number");
	        deriveFieldName(error, column, fieldName);
	        getErrorListener().onDecoderDataError(error);
		    return null;
	    }
	}
	
	protected BigDecimal getBigDecimal(AbstractSegment segment, int column, int rep) 
		throws NumberFormatException, HL7Exception
	{
	    return getBigDecimal(segment,  column,  rep, null);
	}
  
	protected BigDecimal getBigDecimal(NM field)
    	throws NumberFormatException 
    {
		return new BigDecimal(field.getValue().toString());
    }	

	protected Long getLong(NM field) {
	    return new Long(field.getValue().toString());
	}
	
	protected Long getLong(Varies field) 
		throws NumberFormatException 
	{
		return new Long(field.getData().toString());
	}
	
    /**
     * Returns a String from a specific repetition of a Field object at the specified location in the segment. 
	 * @param segment
	 * @param column - field number location in the segment
	 * @param rep - repetition index number of repeating field 
	 */ 	
	protected String getString (AbstractSegment segment, int column, int rep) 
		throws HL7Exception 
	{
	    return HL7Helper.getStringField(segment, column, rep);
	}
	
    /**
     * Returns a concatenated String from an array of all repeating Field objects at the specified location in the segment. 
     * @param segment
     * @param column - field number location in the segment
     */
	protected String getString(AbstractSegment segment, int column) throws HL7Exception {
		return HL7Helper.getStringArray(segment, column);
	}
	
	protected Date getDate(AbstractSegment segment, String rawDate, String providedName) 
		throws HL7Exception
	{
	    try {
	        return stringToDate(rawDate); 
	    }
	    catch (MessagingException e) {
	        throw new HL7Exception(e);
	    }
	}
	
	protected Date getDate(AbstractSegment segment, int column, int rep) 
		throws HL7Exception
	{
	    return getDate(segment, column, rep, null);
	}
	

	protected Date getDate(AbstractSegment segment, int column, int rep, String providedName)  
				throws HL7Exception {
	    String rawDate = null;
	    try {
	        rawDate = getString(segment, column, rep);
	        return stringToDate(rawDate); 
	    }
	    catch (Exception e) {
	        HL7DecoderDataError error = new HL7DecoderDataError();
	        error.setErrorKey(ValidationError.INVALID_DATE_ERROR);
	        error.setComponentNumber(rep);
	        error.setFieldNumber(column); 
	        error.setValue(rawDate);
	        error.setStructure(segment); 
	        deriveFieldName(error, column, providedName);
	        error.setException(e);  
	        getErrorListener().onDecoderDataError(error);
		    return null;
	    }

	}
	
	public String stringFromGenericComposite(AbstractSegment segment, 
	                                         int column, 
	                                         int rep, 
	                                         int componentNumber) 
				throws HL7Exception {
	   
		Type aType = ((Varies)segment.getField(column, rep)).getData();
		
		if(aType instanceof GenericComposite) {
			GenericComposite composite = (GenericComposite)aType;
			Type[] components = composite.getComponents();
			return ((Varies)components[componentNumber]).getData().toString();
		}
		return null;
 
	}
	public String stringFromCodedEntry(AbstractSegment segment, 
	                                         int column, 
	                                         int rep, 
	                                         int componentNumber) 
				throws HL7Exception {
	
		Type aType = ((Varies)segment.getField(column, rep)).getData();
		
		if(aType instanceof CE) {
			CE codedEntry = (CE)aType;
			Type[] components = codedEntry.getComponents();
			return ((ST)components[componentNumber]).getValue();
		}
		return null;
	
	}
	
	public Varies getVaries(AbstractSegment segment, int column, int rep) throws HL7Exception {
	    return (Varies)segment.getField(column,rep);
	}
	
	public String getExpectedHL7Version() {
	    return DEFAULT_EXPECTED_HL7_VERSION;  // default for almost all of the decoders. 
	}
     
    public IHL7ParseEventListener getEventListener() {
        return eventListener;
    }
    
    public IHL7ParseErrorListener getErrorListener() {
        return errorListener;
    }
    
    
    /**
     * subclasses can override to set the field name on the error 
     * @param error
     * @param fieldNumber
     */
    protected void deriveFieldName(HL7DecoderDataError error, int fieldNumber, String providedName) {
        error.setFieldName(providedName);     
    }
    
    protected void setErrorListener(IHL7ParseErrorListener errorListener) {
        this.errorListener = errorListener;
    }
    
    protected void setEventListener(IHL7ParseEventListener eventListener) {
        this.eventListener = eventListener;
    }
    
    /**
     * @return Returns the classOverrideListener.
     */
    public IHL7MessageClassOverrideListener getClassOverrideListener()
    {
        return classOverrideListener;
    }
    
    /**
     * @param classOverrideListener The classOverrideListener to set.
     */
    public void setClassOverrideListener(
            IHL7MessageClassOverrideListener classOverrideListener)
    {
        this.classOverrideListener = classOverrideListener;
    }
    
    public HL7MessageMetadata getMetaData() {
        if (metaData == null) {
            metaData = new HL7MessageMetadata();
        }
        return metaData;
    }
    
    public void setMetaData(HL7MessageMetadata metaData) {
        this.metaData = metaData;
    }
}
